/*
 * SPDX license identifier: MPL-2.0
 *
 * Copyright (C) 2023 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology and Bosch.
 *
 * This file is part of COVESA Project DLT - Diagnostic Log and Trace.
 *
 * This Source Code Form is subject to the terms of the
 * Mozilla Public License (MPL), v. 2.0.
 * If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * For further information see https://covesa.global.
 */

/*!
 * \author
 * Le Van Khanh <Khanh.LeVan@vn.bosch.com>
 *
 * \copyright Copyright © 2023 Advanced Driver Information Technology. \n
 * License MPL-2.0: Mozilla Public License version 2.0 http://mozilla.org/MPL/2.0/.
 *
 * \file dlt_daemon_statistic.h
 */

/*******************************************************************************
**                                                                            **
**  SRC-MODULE: dlt_daemon_statistic.h                                        **
**                                                                            **
**  TARGET    : linux                                                         **
**                                                                            **
**  PROJECT   : DLT                                                           **
**                                                                            **
**  AUTHOR    : Le Van Khanh <Khanh.LeVan@vn.bosch.com>                       **
**                                                                            **
**  PURPOSE   :                                                               **
**                                                                            **
**  REMARKS   :                                                               **
**                                                                            **
**  PLATFORM DEPENDANT [yes/no]: yes                                          **
**                                                                            **
**  TO BE CHANGED BY USER [yes/no]: no                                        **
**                                                                            **
*******************************************************************************/

/*******************************************************************************
**                      Author Identity                                       **
********************************************************************************
**                                                                            **
** Initials     Name                       Company                            **
** --------     -------------------------  ---------------------------------- **
**  cl          Le Van Khanh               Bosch                              **
*******************************************************************************/

#ifndef DLT_DAEMON_STATISTIC_H
#define DLT_DAEMON_STATISTIC_H

#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include "syslog.h"
#include "dlt_daemon_common.h"
#include "dlt-daemon.h"
#include "dlt_daemon_common_cfg.h"


/**
 * Set internal statistics dlt-daemon reference
 * @param daemon pointer to dlt daemon structure
 * @param daemon_local pointer to dlt daemon local structure
 */
void dlt_daemon_statistic_set_daemon_internal(DltDaemon *daemon, DltDaemonLocal *daemon_local);

/**
 * @brief Set the output filename for statistics CSV export
 *
 * This function configures the CSV filename where statistics will be written.
 * It validates the filename length and safely copies it to the internal buffer.
 *
 * @param filename  Path to the CSV output file (must not be NULL)
 * @return 0 on success, -1 on error
 */
int dlt_daemon_statistic_set_filename(const char *filename);

/**
 * @brief Print statistics information to log and CSV file
 *
 * This is the main entry point for exporting DLT statistics. It calculates the
 * duration between start and end timestamps, then exports statistics to both
 * the DLT log and CSV file (if configured).
 * @param user_list             Registered users list
 * @param m_start               Monotonic time start
 * @param m_end                 Monotonic time end
 * @param s_start               System time start
 * @param s_end                 System time end
 * @return 0 on success, -1 on error
 */
int dlt_daemon_statistic_print(DltDaemonRegisteredUsers *user_list,
                                struct timespec m_start, struct timespec m_end,
                                struct timespec s_start, struct timespec s_end);

/**
 * @brief Reset statistics information
 *
 * This function resets all statistical counters (message count and size) for every
 * context across all applications. It iterates through all registered applications
 * and their contexts, setting both stat_cnt and stat_size to zero for each log level.
 *
 * @param user_list  Pointer to registered users list containing applications and contexts
 * @return 0 on success, -1 on error
 */
int dlt_daemon_statistic_reset(DltDaemonRegisteredUsers *user_list);
#endif
